/**
* \file: agwcl.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: sdc-ecm-engine
*
* \brief: Automotive Gateway Convenience Library
*
* \author: Kirill Marinushkin (kmarinushkin@de.adit-jv.com)
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#ifndef AGWCL_INCLUDE_AGWCL_H_
#define AGWCL_INCLUDE_AGWCL_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <sdc.h>

/**
 * \brief Automotive Gateway Convenience Library error codes
 */
typedef enum {
    AGWCL_OK = 0,
    SDC_ERR_BASE = 0,
    /* if error is in range 0 - 0x100 - it is an SDC error code */
    AGWCL_ERR_BASE = 0x100,
    /* if error is 0x100 and larger - it is an AGWCL error code */

    AGWCL_NO_MEM = AGWCL_ERR_BASE,
    AGWCL_INVALID_INPUT,
    AGWCL_ERR_FLAG_KEY_ALG,
    AGWCL_ERR_FLAG_HASH_ALG,
    AGWCL_ERR_FLAG_FORMAT,
    AGWCL_ERR_FLAG_ENCODING,
    AGWCL_ERR_CONF_FILE_NOT_ACCESSIBLE,
    AGWCL_ERR_CONF_PARAM_NOT_FOUND,
    AGWCL_ERR_WRAPPED_FILE_NOT_ACCESSIBLE,
    AGWCL_ERR_CERT_FORMAT,
    AGWCL_ERR_PKCS7_FORMAT,
    AGWCL_ERR_DECRYPT,
    AGWCL_ERR_CREATE_PKCS7_ATTRIBUTE,
    AGWCL_ERR_FLAG_INPUT_ENCODING,
    AGWCL_ERR_UNKNOWN = -1,
} agwcl_error_t;

/**
 * \brief Command flags for sign_with_device_key
 */
/* key algorithm */
#define AGWCL_F_SIG_RSA         0x01
/* hash algorithm */
#define AGWCL_F_SIG_MD5         0x10
#define AGWCL_F_SIG_SHA1        0x20
#define AGWCL_F_SIG_SHA224      0x30
#define AGWCL_F_SIG_SHA256      0x40
#define AGWCL_F_SIG_SHA384      0x50
#define AGWCL_F_SIG_SHA512      0x60
/* output signature format */
#define AGWCL_F_SIG_PKCS1       0x200
#define AGWCL_F_SIG_PKCS7       0x300
/* output signature encoding */
#define AGWCL_F_SIG_DER         0x2000
#define AGWCL_F_SIG_SMIME       0x4000

#define AGWCL_F_SIG_MSG                 0x00000
#define AGWCL_F_SIG_PRECOMPUTED_HASH    0x10000

/**
 * \brief Command flags for decrypt_with_device_key
 */
/* key algorithm */
#define AGWCL_F_DEC_RSA         0x01
/* input encrypted data format */
#define AGWCL_F_DEC_PLAIN       0x100
#define AGWCL_F_DEC_PKCS7       0x300
/* input encrypted data encoding */
#define AGWCL_F_DEC_NOENC       0x1000
#define AGWCL_F_DEC_SMIME       0x4000

/**
 * \brief Command flags for get_client_cert
 */
/* output certificate encoding */
#define AGWCL_F_CERT_PEM         0x3000

/**
 * \brief Sign with the device specific key
 *
 * This method will generate a PKCS#7: version 1.5 (signed data) signature
 * on the given data. The device specific private key will be used for the
 * signature.
 *
 * \param[in] data Data to be signed by the device specific key
 * \param[in] data_len The length of the data to be signed by the
 *                     device specific key
 * \param[out] signature Pointer to a pointer to which the resulting signature
 *                       is written. The function will allocate the memory for
 *                       the signature which has to be freed by the user of the
 *                       function.
 * \param[out] signature_len Pointer to length of the signature.
 *                           Set by the method
 * \param[in] flags A combination of AGWCL_F_SIG_<...> flags
 * \return AGWCL error code
 */
agwcl_error_t sign_with_device_key(unsigned char *data, size_t data_len,
                         unsigned char **signature,  size_t *signature_len,
                         uint64_t flags);

/**
 * \brief Decrypt with the device specific key
 *
 * This method will take a PKCS#7: version 1.5 (encrypted data) data and
 * decrypt it with the device specific key.
 *
 * \param[in] cipherdata Data to be decrypted by the device specific key
 * \param[in] cipherdata_len The length of the data to be decrypted by the
 *                           device specific key
 * \param[out] plaintext Pointer to a pointer to which the resulting plain-text
 *                       is written. The function will allocate the memory for
 *                       the signature which has to be freed by the user of the
 *                       function.
 * \param[out] plaintext_len Pointer to length of the plaintext.
 *                           Set by the method
 * \param[in] flags A combination of AGWCL_F_DEC_<...> flags
 * \return AGWCL error code
 */
agwcl_error_t decrypt_with_device_key(unsigned char *cipherdata,
                            size_t cipherdata_len, unsigned char **plaintext,
                            size_t *plaintext_len, uint64_t flags);

/**
 * \brief This method will return the certificate for the device specific key
 *        in PEM format.
 *
 * This method will return a buffer containing the device certificate in PEM
 * encoding. The memory for the buffer is allocated by the function and has to
 * be freed by the user of the function
 *
 * \param[out] certificate Pointer to a pointer to which the retrieve
 *                         certificate is written. The function will allocate
 *                         the memory for the certificate which has to be freed
 *                         by the user of the function
 * \param[out] certificate_len Pointer to length of the certificate.
 *                             Set by the method
 * \param[in] flags A combination of AGWCL_F_CERT_<...> flags
 * \return AGWCL error code
 */
agwcl_error_t get_client_cert(unsigned char **certificate,
                            size_t *certificate_len, uint64_t flags);

/**
 * \brief The function will securely free the memory retrieved by function in
 *        this API.
 *
 * This method will securely free the memory retrieved by the functions in this
 * API. This should be used for data that should kept secret, e.g. plaintext.
 * This function should currently only be used for the plaintext parameter of
 * the decrypt_with_device_key function
 *
 * \param[in] secret_data Pointer to secret data to be freed
 * \param[in] secret_data_len The length of the data to be securely freed
 */
void free_secret_data(void *secret_data, size_t secret_data_len);

#ifdef __cplusplus
}
#endif

#endif /* AGWCL_INCLUDE_AGWCL_H_ */
